/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.starlink.ttools.plot2;

import java.util.logging.Level;
import java.util.logging.Logger;
import uk.ac.starlink.ttools.func.Maths;
import uk.ac.starlink.ttools.plot.Range;
import uk.ac.starlink.ttools.plot2.Equality;
import uk.ac.starlink.ttools.plot2.PlotUtil;
import uk.ac.starlink.ttools.plot2.Scaler;
import uk.ac.starlink.ttools.plot2.Subrange;

@Equality
public abstract class Scaling {
    private final String name_;
    private final String description_;
    private final boolean isLogLike_;
    public static final Scaling LINEAR = Scaling.createLinearScaling("Linear");
    public static final Scaling LOG = Scaling.createLogScaling("Log");
    public static final Scaling SQRT = Scaling.createSqrtScaling("Sqrt");
    public static final Scaling SQUARE = Scaling.createSquareScaling("Square");
    public static final Scaling AUTO = Scaling.createAutoScaling("Auto");
    private static final double AUTO_DELTA = 0.0625;
    private static final double UNSCALE_TOL = 1.0E-4;
    private static final double UNSCALE_MAXIT = 50.0;
    private static final Scaling[] STRETCHES = new Scaling[]{LOG, LINEAR, SQRT, SQUARE};
    private static final Logger logger_ = Logger.getLogger("uk.ac.starlink.ttools.plot2.layer");

    protected Scaling(String name, String description, boolean isLogLike) {
        this.name_ = name;
        this.description_ = description;
        this.isLogLike_ = isLogLike;
    }

    public String getName() {
        return this.name_;
    }

    public String getDescription() {
        return this.description_;
    }

    public boolean isLogLike() {
        return this.isLogLike_;
    }

    public abstract Scaler createScaler(double var1, double var3);

    public String toString() {
        return this.getName();
    }

    public static final Scaling[] getStretchOptions() {
        return (Scaling[])STRETCHES.clone();
    }

    public static Scaling subrangeScaling(Scaling scaling, Subrange subrange) {
        return Subrange.isIdentity(subrange) ? scaling : new SubrangeScaling(scaling, subrange);
    }

    public static Scaler createRangeScaler(Scaling scaling, Range range) {
        double[] bounds = range.getFiniteBounds(scaling.isLogLike());
        return scaling.createScaler(bounds[0], bounds[1]);
    }

    private static Scaling createLinearScaling(String name) {
        return new ClippedScaling(name, "Linear scaling", false){

            @Override
            public Scaler createClippedScaler(final double lo, double hi) {
                final double scale = 1.0 / (hi - lo);
                return new Scaler(){

                    @Override
                    public double scaleValue(double val) {
                        return (val - lo) * scale;
                    }
                };
            }
        };
    }

    private static Scaling createLogScaling(String name) {
        return new ClippedScaling(name, "Logarithmic scaling", true){

            @Override
            public Scaler createClippedScaler(double lo, double hi) {
                double xlo;
                if (lo > 0.0) {
                    xlo = lo;
                } else if (hi > 1.0) {
                    xlo = 1.0;
                } else if (hi > 0.0) {
                    xlo = hi * 0.001;
                } else {
                    xlo = 0.1;
                    hi = 10.0;
                }
                final double base1 = 1.0 / xlo;
                final double scale = 1.0 / (Math.log(hi) - Math.log(xlo));
                return new Scaler(){

                    @Override
                    public double scaleValue(double val) {
                        return val > 0.0 ? Math.log(val * base1) * scale : 0.0;
                    }
                };
            }
        };
    }

    private static Scaling createAutoScaling(String name) {
        final Scaling asinh = Scaling.createAsinhScaling("Asinh-auto", 0.0625);
        double minSpan = 17.0;
        return new Scaling(name, "asinh-based scaling with default parameters", false){

            @Override
            public Scaler createScaler(double lo, double hi) {
                return asinh.createScaler(lo, Math.max(hi, lo + 17.0));
            }
        };
    }

    private static Scaling createSqrtScaling(String name) {
        return new ReScaling(name, "Square root scaling", LINEAR, new Scaler(){

            @Override
            public double scaleValue(double val) {
                return Math.sqrt(val);
            }
        });
    }

    private static Scaling createSquareScaling(String name) {
        return new ReScaling(name, "Square scaling", LINEAR, new Scaler(){

            @Override
            public double scaleValue(double val) {
                return val * val;
            }
        });
    }

    public static Scaling createAsinhScaling(String name, double delta) {
        String descrip = "asinh-based scaling in which a unit difference at the bottom of the input scale translates to a difference of " + delta + " " + "in the output";
        return new AsinhScaling(name, descrip, delta);
    }

    static double unscale(Scaler scaler, double lo, double hi, double frac) {
        return Scaling.unscaleBisect(scaler, lo, hi, frac);
    }

    private static double unscaleBisect(Scaler scaler, double lo, double hi, double frac) {
        boolean n = false;
        double dLo = lo;
        double dHi = hi;
        double fTarget = frac;
        double fLo = scaler.scaleValue(dLo);
        double fHi = scaler.scaleValue(dHi);
        if (fLo == frac) {
            return dLo;
        }
        if (fHi == frac) {
            return dHi;
        }
        int i = 0;
        while ((double)i < 50.0) {
            double dMid = 0.5 * (dLo + dHi);
            double fMid = scaler.scaleValue(dMid);
            assert (fMid >= 0.0 && fMid <= 1.0);
            assert (dMid >= lo && dMid <= hi);
            if (Math.abs(fMid - fTarget) < 1.0E-4) {
                return dMid;
            }
            if ((fLo - fTarget) / (fMid - fTarget) > 0.0) {
                fLo = fMid;
                dLo = dMid;
            } else {
                fHi = fMid;
                dHi = dMid;
            }
            ++i;
        }
        Level level = Level.INFO;
        if (logger_.isLoggable(level)) {
            logger_.info("Unscale did not converge after 50.0 iterations");
        }
        return lo + frac * (hi - lo);
    }

    private static double unscaleSecant(Scaler scaler, double lo, double hi, double frac) {
        double fTarget = frac;
        double d1 = lo;
        double d2 = hi;
        double f1 = scaler.scaleValue(d1);
        double f2 = scaler.scaleValue(d2);
        int i = 0;
        while ((double)i < 50.0) {
            double d0 = (d2 * (f1 - fTarget) - d1 * (f2 - fTarget)) / (f1 - fTarget - (f2 - fTarget));
            double f0 = scaler.scaleValue(d0);
            if (Math.abs(f0 - fTarget) < 1.0E-4) {
                return d0;
            }
            d2 = d1;
            f2 = f1;
            d1 = d0;
            f1 = f0;
            ++i;
        }
        Level level = Level.INFO;
        if (logger_.isLoggable(level)) {
            logger_.info("Unscale did not converge after 50.0 iterations");
        }
        return lo + frac * (hi - lo);
    }

    private static class AsinhScaler
    implements Scaler {
        private final double u_;
        private final double v_;

        AsinhScaler(double delta, double max) {
            this(AsinhScaler.calcCoeffs(delta, max));
            assert (this.scaleValue(0.0) == 0.0);
            assert (PlotUtil.approxEquals(delta, this.scaleValue(1.0)));
            assert (this.scaleValue(max) <= 1.0);
        }

        private AsinhScaler(double[] coeffs) {
            this.u_ = coeffs[0];
            this.v_ = coeffs[1];
        }

        @Override
        public double scaleValue(double c) {
            return this.u_ * Maths.asinh(this.v_ * c);
        }

        private static double[] calcCoeffs(double delta, double max) {
            double v0 = 1.0;
            boolean done = false;
            while (!done) {
                double[] derivs = AsinhScaler.calcDerivsV(v0, max, delta);
                double v1 = v0 - derivs[0] / derivs[1];
                done = Math.abs(v1 - v0) < 1.0E-14;
                v0 = v1;
            }
            double v = v0;
            double u = 1.0 / Maths.asinh(v * max);
            return new double[]{u, v};
        }

        private static double[] calcDerivsV(double v, double max, double delta) {
            double d0 = Maths.sinh(delta * Maths.asinh(v * max)) - v;
            double d1 = Maths.cosh(delta * Maths.asinh(v * max)) * delta * max / Math.hypot(v * max, 1.0) - 1.0;
            return new double[]{d0, d1};
        }
    }

    private static abstract class ClippedScaling
    extends Scaling {
        protected ClippedScaling(String name, String description, boolean isLogLike) {
            super(name, description, isLogLike);
        }

        protected abstract Scaler createClippedScaler(double var1, double var3);

        @Override
        public Scaler createScaler(final double lo, final double hi) {
            if (lo < hi) {
                final Scaler clipScaler = this.createClippedScaler(lo, hi);
                final double loOut = clipScaler.scaleValue(lo);
                final double hiOut = clipScaler.scaleValue(hi);
                return new Scaler(){

                    @Override
                    public double scaleValue(double val) {
                        if (val <= lo) {
                            return loOut;
                        }
                        if (val >= hi) {
                            return hiOut;
                        }
                        if (Double.isNaN(val)) {
                            return Double.NaN;
                        }
                        return clipScaler.scaleValue(val);
                    }
                };
            }
            if (lo == hi) {
                final double midVal = lo;
                double elo = midVal * 0.999;
                double ehi = midVal * 1.001;
                Scaler clipScaler = this.createClippedScaler(elo, ehi);
                final double loOut = ClippedScaling.clipUnit(clipScaler.scaleValue(elo));
                final double hiOut = ClippedScaling.clipUnit(clipScaler.scaleValue(ehi));
                double midOut = clipScaler.scaleValue(midVal);
                return new Scaler(){

                    @Override
                    public double scaleValue(double val) {
                        if (val < midVal) {
                            return loOut;
                        }
                        if (val > midVal) {
                            return hiOut;
                        }
                        if (val == midVal) {
                            return midVal;
                        }
                        return Double.NaN;
                    }
                };
            }
            throw new IllegalArgumentException("! " + lo + " < " + hi);
        }

        private static double clipUnit(double val) {
            return Math.min(1.0, Math.max(0.0, val));
        }
    }

    private static class SubrangeScaling
    extends Scaling {
        private final Scaling baseScaling_;
        private final Subrange subrange_;

        SubrangeScaling(Scaling baseScaling, Subrange subrange) {
            super(baseScaling.getName() + "-sub", baseScaling.getDescription() + ", subrange: " + subrange, baseScaling.isLogLike());
            this.baseScaling_ = baseScaling;
            this.subrange_ = subrange;
        }

        @Override
        public Scaler createScaler(double lo, double hi) {
            double subHi;
            Scaler fullScaler = this.baseScaling_.createScaler(lo, hi);
            double subLo = SubrangeScaling.unscale(fullScaler, lo, hi, this.subrange_.getLow());
            return subLo < (subHi = SubrangeScaling.unscale(fullScaler, lo, hi, this.subrange_.getHigh())) ? this.baseScaling_.createScaler(subLo, subHi) : this.baseScaling_.createScaler(subHi, subLo);
        }

        public int hashCode() {
            int code = 688923;
            code = 23 * code + this.baseScaling_.hashCode();
            code = 23 * code + this.subrange_.hashCode();
            return code;
        }

        public boolean equals(Object o) {
            if (o instanceof SubrangeScaling) {
                SubrangeScaling other = (SubrangeScaling)o;
                return this.baseScaling_.equals(other.baseScaling_) && this.subrange_.equals(other.subrange_);
            }
            return false;
        }
    }

    private static class ReScaling
    extends Scaling {
        private final Scaling baseScaling_;
        private final Scaler rescaler_;

        public ReScaling(String name, String description, Scaling baseScaling, Scaler rescaler) {
            super(name, description, baseScaling.isLogLike());
            this.baseScaling_ = baseScaling;
            this.rescaler_ = rescaler;
        }

        @Override
        public Scaler createScaler(double lo, double hi) {
            final Scaler baseScaler = this.baseScaling_.createScaler(lo, hi);
            return new Scaler(){

                @Override
                public double scaleValue(double val) {
                    return ReScaling.this.rescaler_.scaleValue(baseScaler.scaleValue(val));
                }
            };
        }

        public int hashCode() {
            int code = 33441;
            code = 23 * code + this.baseScaling_.hashCode();
            code = 23 * code + this.rescaler_.hashCode();
            return code;
        }

        public boolean equals(Object o) {
            if (o instanceof ReScaling) {
                ReScaling other = (ReScaling)o;
                return this.baseScaling_ == other.baseScaling_ && this.rescaler_ == other.rescaler_;
            }
            return false;
        }
    }

    private static class AsinhScaling
    extends ClippedScaling {
        private final double delta_;

        AsinhScaling(String name, String description, double delta) {
            super(name, description, false);
            this.delta_ = delta;
        }

        @Override
        public Scaler createClippedScaler(final double lo, double hi) {
            final AsinhScaler zScaler = new AsinhScaler(this.delta_, hi - lo);
            return new Scaler(){

                @Override
                public double scaleValue(double val) {
                    return zScaler.scaleValue(val - lo);
                }
            };
        }

        public int hashCode() {
            int code = 79982;
            code = 23 * code + Float.floatToIntBits((float)this.delta_);
            return code;
        }

        public boolean equals(Object o) {
            return o instanceof AsinhScaling && this.delta_ == ((AsinhScaling)o).delta_;
        }
    }
}

