/*
 * Decompiled with CFR 0.152.
 */
package uk.ac.starlink.ttools.task;

import gov.fnal.eag.healpix.PixTools;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.logging.Logger;
import javax.vecmath.Vector3d;
import uk.ac.starlink.table.ColumnInfo;
import uk.ac.starlink.table.DescribedValue;
import uk.ac.starlink.table.StarTable;
import uk.ac.starlink.table.Tables;

public class PixSampler {
    private final StarTable pixTable_;
    private final long nside_;
    private final boolean nested_;
    private final int ncol_;
    private final PixTools pixTools_;
    private static final Logger logger_ = Logger.getLogger("uk.ac.starlink.ttools.calc");
    public static final StatMode POINT_MODE = new PointStatMode("Point");
    public static final StatMode MEAN_MODE = new MeanStatMode("Mean");

    public PixSampler(StarTable pixTable, long nside, boolean nested) throws IOException {
        long requireNrow;
        if (!pixTable.isRandom()) {
            throw new IOException("Pixel data not random access");
        }
        this.pixTools_ = PixTools.getInstance();
        long hasNrow = pixTable.getRowCount();
        if (hasNrow != (requireNrow = this.pixTools_.Nside2Npix(nside))) {
            throw new IOException("Wrong number of rows for nside " + nside + " (" + hasNrow + "!=" + requireNrow + ")");
        }
        this.pixTable_ = pixTable;
        this.nside_ = nside;
        this.nested_ = nested;
        this.ncol_ = pixTable.getColumnCount();
    }

    public Object sampleValue(int icol, double alphaDeg, double deltaDeg, double radiusDeg, StatMode statMode) throws IOException {
        if (alphaDeg >= -360.0 && alphaDeg <= 360.0 && deltaDeg >= -90.0 && deltaDeg <= 90.0) {
            Object[] samples;
            if (statMode.isPoint()) {
                long irow = this.getPixIndex(alphaDeg, deltaDeg);
                samples = new Object[]{this.pixTable_.getCell(irow, icol)};
            } else {
                long[] irows = this.getPixIndices(alphaDeg, deltaDeg, radiusDeg);
                int nr = irows.length;
                samples = new Object[nr];
                for (int ir = 0; ir < nr; ++ir) {
                    samples[ir] = this.pixTable_.getCell((long)ir, icol);
                }
            }
            return statMode.getResult(samples);
        }
        return null;
    }

    public Object[] sampleValues(double alphaDeg, double deltaDeg, double radiusDeg, StatMode statMode) throws IOException {
        if (alphaDeg >= -360.0 && alphaDeg <= 360.0 && deltaDeg >= -90.0 && deltaDeg <= 90.0) {
            Object[][] samples;
            if (statMode.isPoint()) {
                long irow = this.getPixIndex(alphaDeg, deltaDeg);
                samples = new Object[this.ncol_][1];
                Object[] row = this.pixTable_.getRow(irow);
                for (int icol = 0; icol < this.ncol_; ++icol) {
                    samples[icol][0] = row[icol];
                }
            } else {
                long[] irows = this.getPixIndices(alphaDeg, deltaDeg, radiusDeg);
                int nr = irows.length;
                samples = new Object[this.ncol_][nr];
                for (int ir = 0; ir < nr; ++ir) {
                    Object[] row = this.pixTable_.getRow(irows[ir]);
                    for (int icol = 0; icol < this.ncol_; ++icol) {
                        samples[icol][ir] = row[icol];
                    }
                }
            }
            Object[] outRow = new Object[this.ncol_];
            for (int ic = 0; ic < this.ncol_; ++ic) {
                outRow[ic] = statMode.getResult(samples[ic]);
            }
            return outRow;
        }
        return new Object[this.ncol_];
    }

    public ColumnInfo[] getValueInfos(StatMode statMode) {
        int ncol = this.pixTable_.getColumnCount();
        ColumnInfo[] infos = new ColumnInfo[this.ncol_];
        for (int icol = 0; icol < this.ncol_; ++icol) {
            infos[icol] = statMode.getResultInfo(this.pixTable_.getColumnInfo(icol));
        }
        return infos;
    }

    private long getPixIndex(double alphaDeg, double deltaDeg) {
        double phiRad = PixSampler.alphaToPhi(alphaDeg);
        double thetaRad = PixSampler.deltaToTheta(deltaDeg);
        return this.nested_ ? this.pixTools_.ang2pix_nest(this.nside_, thetaRad, phiRad) : this.pixTools_.ang2pix_ring(this.nside_, thetaRad, phiRad);
    }

    private long[] getPixIndices(double alphaDeg, double deltaDeg, double radiusDeg) {
        double phiRad = PixSampler.alphaToPhi(alphaDeg);
        double thetaRad = PixSampler.deltaToTheta(deltaDeg);
        double radiusRad = radiusDeg * Math.PI / 180.0;
        Vector3d vec = this.pixTools_.Ang2Vec(thetaRad, phiRad);
        ArrayList pixList = this.pixTools_.query_disc(this.nside_, vec, radiusRad, this.nested_ ? 1 : 0, 0);
        long[] pixes = new long[pixList.size()];
        int ip = 0;
        Iterator it = pixList.iterator();
        while (it.hasNext()) {
            pixes[ip++] = ((Number)it.next()).longValue();
        }
        assert (ip == pixes.length);
        return pixes;
    }

    private static double alphaToPhi(double alphaDeg) {
        double alphaRad = alphaDeg * Math.PI / 180.0;
        return alphaRad;
    }

    private static double deltaToTheta(double deltaDeg) {
        double deltaRad = deltaDeg * Math.PI / 180.0;
        return 1.5707963267948966 - deltaRad;
    }

    public static PixSampler createPixSampler(StarTable pixTable) throws IOException {
        boolean nested;
        if (!pixTable.isRandom()) {
            throw new IOException("Pixel data not random access");
        }
        int nside = PixSampler.inferNside(pixTable);
        Boolean isNested = PixSampler.inferNested(pixTable);
        if (isNested == null) {
            logger_.warning("Cannot determine HEALPix ordering scheme - assuming nested");
            nested = true;
        } else {
            nested = isNested;
        }
        return new PixSampler(pixTable, nside, nested);
    }

    public static Boolean inferNested(StarTable pixTable) {
        String orderKey = "ORDERING";
        String ordering = PixSampler.getStringParam(pixTable, orderKey);
        if (ordering != null) {
            String hval = "Header " + orderKey + "=\"" + ordering.trim() + "\"";
            if (ordering.toUpperCase().startsWith("NEST")) {
                logger_.info(hval + " - inferring NESTED HEALPix ordering");
                return Boolean.TRUE;
            }
            if (ordering.toUpperCase().startsWith("RING")) {
                logger_.info(hval + " - inferring RING HEALPix ordering");
                return Boolean.FALSE;
            }
            logger_.warning(hval + " - unknown value");
            return null;
        }
        return null;
    }

    public static int inferNside(StarTable pixTable) throws IOException {
        long nside;
        if (!pixTable.isRandom()) {
            throw new IOException("Pixel data not random access");
        }
        long nrow = pixTable.getRowCount();
        try {
            nside = PixTools.getInstance().Npix2Nside(nrow);
        }
        catch (RuntimeException e) {
            nside = -1L;
        }
        if (nside <= 0L) {
            throw new IOException("Unsuitable number of rows for all-sky HEALPix map (" + nrow + ")");
        }
        String pixtype = PixSampler.getStringParam(pixTable, "PIXTYPE");
        boolean dHealpix = "HEALPIX".equalsIgnoreCase(pixtype);
        double dNside = PixSampler.getNumericParam(pixTable, "NSIDE");
        if (dNside >= 0.0 && dNside != (double)nside) {
            String msg = "NSIDE mismatch: declared (" + dNside + ")" + " != count (" + nside + ")";
            if (dHealpix) {
                throw new IOException(msg);
            }
            logger_.warning(msg);
        }
        return Tables.checkedLongToInt((long)nside);
    }

    private static String getStringParam(StarTable table, String key) {
        Object value;
        DescribedValue dval = table.getParameterByName(key);
        if (dval != null && (value = dval.getValue()) instanceof String) {
            return (String)value;
        }
        return null;
    }

    private static double getNumericParam(StarTable table, String key) {
        Object value;
        DescribedValue dval = table.getParameterByName(key);
        if (dval != null && (value = dval.getValue()) instanceof Number) {
            return ((Number)value).doubleValue();
        }
        return Double.NaN;
    }

    private static class MeanStatMode
    extends AbstractStatMode {
        MeanStatMode(String name) {
            super(name, false);
        }

        @Override
        public ColumnInfo getResultInfo(ColumnInfo baseInfo) {
            String baseDesc = baseInfo.getDescription();
            String desc = baseDesc == null || baseDesc.trim().length() == 0 ? "Mean value" : baseDesc + ", spatial mean";
            return new ColumnInfo(baseInfo.getName(), Double.class, desc);
        }

        @Override
        public Object getResult(Object[] values) {
            double sum = 0.0;
            double count = 0.0;
            for (int i = 0; i < values.length; ++i) {
                double dval;
                Object val = values[i];
                if (!(val instanceof Number) || Double.isNaN(dval = ((Number)val).doubleValue())) continue;
                sum += dval;
                count += 1.0;
            }
            return count > 0.0 ? new Double(sum / count) : Double.NaN;
        }
    }

    private static class PointStatMode
    extends AbstractStatMode {
        PointStatMode(String name) {
            super(name, true);
        }

        @Override
        public ColumnInfo getResultInfo(ColumnInfo baseInfo) {
            return baseInfo;
        }

        @Override
        public Object getResult(Object[] values) {
            return values != null && values.length > 0 ? values[0] : null;
        }
    }

    private static abstract class AbstractStatMode
    implements StatMode {
        private final String name_;
        private final boolean isPoint_;

        protected AbstractStatMode(String name, boolean isPoint) {
            this.name_ = name;
            this.isPoint_ = isPoint;
        }

        @Override
        public boolean isPoint() {
            return this.isPoint_;
        }

        public String toString() {
            return this.name_;
        }
    }

    public static interface StatMode {
        public ColumnInfo getResultInfo(ColumnInfo var1);

        public Object getResult(Object[] var1);

        public boolean isPoint();
    }
}

